// Copyright 2022 The Forgotten Server Authors. All rights reserved.
// Use of this source code is governed by the GPL-2.0 License that can be found in the LICENSE file.

#include "otpch.h"
#include "ban.h"
#include "database.h"
#include "databasetasks.h"
#include "tools.h"

#include <fmt/format.h>

// Verifica se um IP pode se conectar (sistema de proteção contra flood)
bool Ban::acceptConnection(uint32_t clientIP)
{
	std::lock_guard<std::recursive_mutex> lockClass(lock);

	uint64_t currentTime = OTSYS_TIME();

	auto it = ipConnectMap.find(clientIP);
	if (it == ipConnectMap.end()) {
		ipConnectMap.emplace(clientIP, ConnectBlock(currentTime, 0, 1));
		return true;
	}

	ConnectBlock& connectBlock = it->second;
	if (connectBlock.blockTime > currentTime) {
		connectBlock.blockTime += 250;
		return false;
	}

	int64_t timeDiff = currentTime - connectBlock.lastAttempt;
	connectBlock.lastAttempt = currentTime;
	if (timeDiff <= 5000) {
		if (++connectBlock.count > 5) {
			connectBlock.count = 0;
			if (timeDiff <= 500) {
				connectBlock.blockTime = currentTime + 3000;
				return false;
			}
		}
	} else {
		connectBlock.count = 1;
	}
	return true;
}

// Verifica se uma conta está banida
bool IOBan::isAccountBanned(uint32_t accountId, BanInfo& banInfo)
{
    Database& db = Database::getInstance();

    DBResult_ptr result = db.storeQuery(fmt::format(
        "SELECT `comment`, `expires_at`, `banned_at`, `banned_by`, "
        "(SELECT `name` FROM `players` WHERE `id` = `banned_by`) AS `name` "
        "FROM `account_bans` WHERE `account_id` = {:d} AND `expires_at` > {}", accountId, time(nullptr)));

    if (!result) {
        return false;
    }

    banInfo.expiresAt = result->getNumber<time_t>("expires_at");
    banInfo.reason = result->getString("comment"); // <-- Agora está pegando o motivo correto!
    banInfo.bannedBy = result->getString("name");
    return true;
}



bool IOBan::isPlayerBanned(uint32_t playerId, BanInfo& banInfo)
{
    Database& db = Database::getInstance();

    DBResult_ptr result = db.storeQuery(fmt::format(
        "SELECT `comment`, `expires_at`, `banned_at`, `banned_by`, "
        "(SELECT `name` FROM `players` WHERE `id` = `banned_by`) AS `name` "
        "FROM `player_bans` WHERE `player_id` = {:d} AND `expires_at` > {}", playerId, time(nullptr)));

    if (!result) {
        return false;
    }

    banInfo.expiresAt = result->getNumber<time_t>("expires_at");
    banInfo.reason = result->getString("comment");
    banInfo.bannedBy = result->getString("name");
    return true;
}


// Adiciona um banimento para uma conta
bool IOBan::addAccountBanishment(uint32_t accountId, time_t banTime, uint32_t reasonId, uint32_t actionId, const std::string& comment, uint32_t bannedBy)
{
    Database& db = Database::getInstance();

    // Copia o comentário e escapa aspas simples (') duplicando elas (' -> '')
    std::string safeComment = comment;
    replaceString(safeComment, "'", "''");

    // Agora executa a query com o comentário seguro
    return db.executeQuery(fmt::format(
        "INSERT INTO `account_bans` (`account_id`, `banned_at`, `expires_at`, `reason`, `action_id`, `banned_by`, `comment`) VALUES "
        "({}, {}, {}, {}, {}, {}, '{}')",
        accountId, time(nullptr), banTime, reasonId, actionId, bannedBy, safeComment
    ));
}

bool IOBan::addPlayerBanishment(uint32_t playerId, time_t banTime, uint32_t reasonId, uint32_t actionId, const std::string& comment, uint32_t bannedBy)
{
    Database& db = Database::getInstance();

    std::string safeComment = comment;
    replaceString(safeComment, "'", "''");

    return db.executeQuery(fmt::format(
        "INSERT INTO `player_bans` (`player_id`, `banned_at`, `expires_at`, `reason`, `action_id`, `banned_by`, `comment`) "
        "VALUES ({0}, {1}, {2}, {3}, {4}, {5}, '{6}') "
        "ON DUPLICATE KEY UPDATE "
        "`banned_at` = {1}, `expires_at` = {2}, `reason` = {3}, `action_id` = {4}, `banned_by` = {5}, `comment` = '{6}'",
        playerId, time(nullptr), banTime, reasonId, actionId, bannedBy, safeComment
    ));
}

bool IOBan::isIpBanned(uint32_t clientIP, BanInfo& banInfo)
{
	if (clientIP == 0) {
		return false;
	}

	Database& db = Database::getInstance();

	DBResult_ptr result = db.storeQuery(fmt::format(
		"SELECT `reason`, `expires_at`, (SELECT `name` FROM `players` WHERE `id` = `banned_by`) AS `name` FROM `ip_bans` WHERE `ip` = {:d} AND `expires_at` > {}",
		clientIP, time(nullptr)
	));

	if (!result) {
		return false;
	}

	banInfo.expiresAt = result->getNumber<time_t>("expires_at");
	banInfo.reason = result->getString("reason");
	banInfo.bannedBy = result->getString("name");
	return true;
}

// Limpa banimentos expirados automaticamente
bool IOBan::clearExpiredBans()
{
	Database& db = Database::getInstance();

	return db.executeQuery("DELETE FROM `account_bans` WHERE `expires_at` <= UNIX_TIMESTAMP()") &&
	       db.executeQuery("DELETE FROM `player_bans` WHERE `expires_at` <= UNIX_TIMESTAMP()") &&
	       db.executeQuery("DELETE FROM `ip_bans` WHERE `expires_at` <= UNIX_TIMESTAMP()");
}

bool IOBan::isPlayerNamelocked(uint32_t playerId)
{
    // Verifica se o jogador está na tabela de "player_namelocks"
    return Database::getInstance().storeQuery(fmt::format(
        "SELECT 1 FROM `player_namelocks` WHERE `player_id` = {}", playerId
    )).get() != nullptr;
}
